-- ============================================
-- 动态功能完整数据库表创建脚本
-- ============================================

-- 1. 用户动态表
CREATE TABLE IF NOT EXISTS `user_dynamics` (
  `dynamic_id` INT(11) NOT NULL AUTO_INCREMENT COMMENT '动态ID',
  `user_id` INT(11) NOT NULL COMMENT '发布者用户ID',
  `content` TEXT NOT NULL COMMENT '动态内容',
  `media_urls` TEXT COMMENT '媒体文件URL数组（JSON格式或逗号分隔）',
  `media_type` TINYINT(1) DEFAULT 1 COMMENT '媒体类型：1-纯文本，2-图片，3-视频',
  `audit_status` TINYINT(1) DEFAULT 0 COMMENT '审核状态：0-待审核，1-审核通过，2-审核拒绝，3-人工审核中',
  `audit_remark` VARCHAR(500) COMMENT '审核备注',
  `audit_time` DATETIME COMMENT '审核时间',
  `audit_admin_id` INT(11) COMMENT '审核管理员ID',
  `like_count` INT(11) DEFAULT 0 COMMENT '点赞数',
  `comment_count` INT(11) DEFAULT 0 COMMENT '评论数',
  `favorite_count` INT(11) DEFAULT 0 COMMENT '收藏数',
  `share_count` INT(11) DEFAULT 0 COMMENT '转发数',
  `view_count` INT(11) DEFAULT 0 COMMENT '浏览数',
  `status` TINYINT(1) DEFAULT 1 COMMENT '状态：0-删除，1-正常，2-隐藏',
  `visibility` TINYINT(1) DEFAULT 1 COMMENT '可见性：1-公开，2-仅粉丝，3-私密',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`dynamic_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_audit_status` (`audit_status`),
  KEY `idx_status` (`status`),
  KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='用户动态表';

-- 2. 动态点赞表
CREATE TABLE IF NOT EXISTS `dynamic_likes` (
  `like_id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '点赞ID',
  `dynamic_id` INT(11) NOT NULL COMMENT '动态ID',
  `user_id` INT(11) NOT NULL COMMENT '点赞用户ID',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '点赞时间',
  PRIMARY KEY (`like_id`),
  UNIQUE KEY `uk_dynamic_user` (`dynamic_id`, `user_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='动态点赞表';

-- 3. 动态收藏表
CREATE TABLE IF NOT EXISTS `dynamic_favorites` (
  `favorite_id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '收藏ID',
  `dynamic_id` INT(11) NOT NULL COMMENT '动态ID',
  `user_id` INT(11) NOT NULL COMMENT '收藏用户ID',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '收藏时间',
  PRIMARY KEY (`favorite_id`),
  UNIQUE KEY `uk_dynamic_user` (`dynamic_id`, `user_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='动态收藏表';

-- 4. 动态评论表
CREATE TABLE IF NOT EXISTS `dynamic_comments` (
  `comment_id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '评论ID',
  `dynamic_id` INT(11) NOT NULL COMMENT '动态ID',
  `user_id` INT(11) NOT NULL COMMENT '评论用户ID',
  `parent_comment_id` BIGINT(20) DEFAULT 0 COMMENT '父评论ID（0表示一级评论）',
  `content` TEXT NOT NULL COMMENT '评论内容',
  `image_urls` TEXT COMMENT '评论图片URL（多张以逗号分隔）',
  `like_count` INT(11) DEFAULT 0 COMMENT '点赞数',
  `reply_count` INT(11) DEFAULT 0 COMMENT '回复数',
  `status` TINYINT(1) DEFAULT 1 COMMENT '状态：0-已删除，1-正常',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '评论时间',
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`comment_id`),
  KEY `idx_dynamic_id` (`dynamic_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_parent_comment_id` (`parent_comment_id`),
  KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='动态评论表';

-- 5. 评论点赞表
CREATE TABLE IF NOT EXISTS `comment_likes` (
  `like_id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '点赞ID',
  `comment_id` BIGINT(20) NOT NULL COMMENT '评论ID',
  `user_id` INT(11) NOT NULL COMMENT '点赞用户ID',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '点赞时间',
  PRIMARY KEY (`like_id`),
  UNIQUE KEY `uk_comment_user` (`comment_id`, `user_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='评论点赞表';

-- 6. 动态举报表（已创建）
CREATE TABLE IF NOT EXISTS `dynamic_reports` (
  `report_id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '举报ID',
  `dynamic_id` BIGINT(20) NOT NULL COMMENT '被举报的动态ID',
  `reporter_id` BIGINT(20) NOT NULL COMMENT '举报人用户ID',
  `report_type` VARCHAR(50) NOT NULL COMMENT '举报原因类型：spam-垃圾广告，porn-色情低俗，violence-暴力违法，attack-人身攻击，fake-虚假信息，plagiarism-抄袭侵权，other-其他',
  `description` TEXT COMMENT '详细描述',
  `screenshots` TEXT COMMENT '举报截图URL，多张以逗号分隔',
  `contact` VARCHAR(100) COMMENT '联系方式',
  `status` TINYINT(1) NOT NULL DEFAULT 0 COMMENT '处理状态：0-待处理，1-处理中，2-已处理，3-已驳回',
  `handle_result` TEXT COMMENT '处理结果说明',
  `handler_id` BIGINT(20) COMMENT '处理人ID',
  `handle_time` DATETIME COMMENT '处理时间',
  `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  `deleted` TINYINT(1) NOT NULL DEFAULT 0 COMMENT '逻辑删除：0-未删除，1-已删除',
  PRIMARY KEY (`report_id`),
  KEY `idx_dynamic_id` (`dynamic_id`),
  KEY `idx_reporter_id` (`reporter_id`),
  KEY `idx_status` (`status`),
  KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='动态举报表';

-- ============================================
-- 插入测试数据（可选）
-- ============================================

-- 插入测试动态数据
INSERT INTO `user_dynamics` (`user_id`, `content`, `media_urls`, `media_type`, `audit_status`, `like_count`, `comment_count`, `visibility`) VALUES
(1, '今天天气真好，心情也很好！💕', NULL, 1, 1, 0, 0, 1),
(1, '分享一组美美的照片~', '["http://localhost:9000/marriage/dynamics/test1.jpg","http://localhost:9000/marriage/dynamics/test2.jpg"]', 2, 1, 0, 0, 1),
(2, '参加了一场很棒的活动，遇到了很多有趣的朋友！', NULL, 1, 1, 0, 0, 1);

-- ============================================
-- 表创建完成
-- ============================================

SELECT '✅ 所有动态功能相关表创建成功！' AS status;
SELECT 
    TABLE_NAME AS '表名',
    TABLE_COMMENT AS '说明',
    TABLE_ROWS AS '记录数'
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = DATABASE() 
AND TABLE_NAME IN ('user_dynamics', 'dynamic_likes', 'dynamic_favorites', 'dynamic_comments', 'comment_likes', 'dynamic_reports')
ORDER BY TABLE_NAME;

